<?php

class FrmWpmlString {

	/**
	 * Get the keys for all form options that can be translated.
	 *
	 * @since 1.05
	 * @param object $form
	 */
	public static function get_form_strings( $form ) {
		if ( is_callable( 'FrmForm::translatable_strings' ) ) {
			return FrmForm::translatable_strings( $form );
		}

		_deprecated_function( __FUNCTION__, '1.05', 'Please update Formidable Forms to the latest version.' );

		$form_keys = array(
			'name',
			'description',
			'submit_value',
			'submit_msg',
			'success_msg',
			'email_subject',
			'email_message',
			'ar_email_subject',
			'ar_email_message',
			'draft_msg',
		);

		// Add edit options.
		if ( $form->editable ) {
			$form_keys[] = 'edit_value';
			$form_keys[] = 'edit_msg';
		}

		return $form_keys;
	}

	/**
	 * The used keys for field options.
	 *
	 * @since 1.05
	 * @param object $field
	 */
	public static function field_translation_keys( $field ) {
		$keys = self::get_field_option_keys_for_translations( $field->type );
		self::remove_unused_keys( $field, $keys );
		return $keys;
	}

	/**
	 * Get the keys for all field options that can be translated
	 *
	 * @since 1.05
	 * @param string $field_type
	 * @return array
	 */
	public static function get_field_option_keys_for_translations( $field_type ) {
		$field_obj = FrmFieldFactory::get_field_type( $field_type );
		if ( is_callable( array( $field_obj, 'translatable_strings' ) ) ) {
			return $field_obj->translatable_strings();
		}

		_deprecated_function( __FUNCTION__, '1.05', 'Please update Formidable Forms to the latest version.' );

		$keys = array(
			'name',
			'description',
			'default_value',
			'required_indicator',
			'invalid',
			'blank',
			'unique_msg',
		);

		switch ( $field_type ) {
			case 'end_divider':
				$keys = array( 'add_label', 'remove_label' );
				break;
			case 'divider':
				$keys = array( 'name', 'description' );
				break;
			case 'break':
				$keys = array( 'name' );
				break;
			case 'date':
				$keys[] = 'locale';
				break;
			case 'email':
			case 'password':
				$keys[] = 'conf_desc';
				break;
			case 'address':
				$keys[] = 'line1_desc';
				$keys[] = 'line2_desc';
				$keys[] = 'city_desc';
				$keys[] = 'state_desc';
				$keys[] = 'zip_desc';
				$keys[] = 'country_desc';
				break;
			case 'toggle':
				$keys[] = 'toggle_on';
				$keys[] = 'toggle_off';
		}

		return $keys;
	}

	/**
	 * Don't include the unique message unless it is needed.
	 *
	 * @since 1.05
	 * @param object $field
	 * @param array  $keys
	 */
	public static function remove_unused_keys( $field, &$keys ) {
		$remove = array();

		$is_unique = ( isset( $field->field_options['unique'] ) && $field->field_options['unique'] );
		if ( ! $is_unique ) {
			$remove[] = array_search( 'unique_msg', $keys );
		}

		if ( ! $field->required ) {
			$remove[] = array_search( 'blank', $keys );
		}

		$remove_if = array(
			'required_indicator' => '*',
			'toggle_on'          => 1,
			'toggle_off'         => 0,
		);
		foreach ( $remove_if as $opt => $value ) {
			$is_default = ( isset( $field->field_options[ $opt ] ) && $field->field_options[ $opt ] == $value );
			if ( $is_default ) {
				$remove[] = array_search( $opt, $keys );
			}
		}

		foreach ( $remove as $remove_key ) {
			if ( $remove_key !== false ) {
				unset( $keys[ $remove_key ] );
			}
		}
	}

	/**
	 * Get the default value for translating
	 * Do not translate default value for certain field types
	 * Do not translate shortcodes
	 *
	 * @since 1.05
	 * @param object $field
	 * @return string
	 */
	public static function get_default_value_for_translation( $field ) {
		if ( self::default_value_needs_translation( $field ) ) {
			$default_value = self::get_default_value( $field );
		} else {
			$default_value = '';
		}

		return $default_value;
	}

	/**
	 * Get the default value for a field
	 *
	 * @since 1.05
	 * @param object $field
	 * @return mixed
	 */
	private static function get_default_value( $field ) {
		if ( FrmField::is_option_true_in_object( $field, 'dyn_default_value' ) ) {
			$default_value = $field->field_options['dyn_default_value'];
		} else if ( isset( $field->default_value ) && $field->default_value ) {
			$default_value = $field->default_value;
		} else {
			$default_value = '';
		}

		return $default_value;
	}

	/**
	 * Check if a specific field option needs translating
	 *
	 * @since 1.05
	 * @param array $values
	 * @param string $key
	 * @return bool
	 */
	public static function field_option_needs_translation( $values, $key ) {
		$needs_translation = true;

		if ( ! isset( $values[ $key ] ) ) {
			$needs_translation = false;
		} else if ( is_array( $values[ $key ] ) && $key !== 'default_value' ) {
			$needs_translation = false;
		} else if ( in_array( $values[ $key ], array( '', '*' ) ) ) {
			$needs_translation = false;
		} else if ( $key == 'default_value' && self::default_value_needs_translation( $values ) === false ) {
			$needs_translation = false;
		}

		return $needs_translation;
	}

	/**
	 * Check if a field needs it default value translated
	 *
	 * @since 1.05
	 * @param object|array $field
	 * @return bool
	 */
	public static function default_value_needs_translation( $field ) {
		if ( is_object( $field ) ) {
			$field_type = $field->type;
		} else {
			$field_type = isset( $field['original_type'] ) ? $field['original_type'] : $field['type'];
		}

		return ! in_array( $field_type, array( 'radio', 'checkbox', 'select', 'scale', 'data' ) );
	}

	public static function form_strings( $form ) {
		if ( ! is_object( $form ) ) {
			$form = FrmForm::getOne( $form );
		}

		$string_data = array();

		self::get_translatable_strings_from_form_settings( $form, $string_data );

		// TODO: maybe create form or field object to cut down on variables
		$is_repeating = false;
		$fields = FrmField::getAll( array( 'fi.form_id' => $form->id ), 'field_order' );
		foreach ( $fields as $field ) {
			if ( FrmField::is_repeating_field( $field ) ) {
				$is_repeating = true;
			}

			self::get_translatable_strings_for_single_field( $field, $form->id, $is_repeating, $string_data );

			if ( $field->type == 'end_divider' && $is_repeating ) {
				$is_repeating = false;
			}
		}

		return $string_data;

	}

	/**
	 * Get the strings that need translation from a form's Settings
	 *
	 * @since 1.05
	 * @param object $form
	 * @param array $string_data
	 */
	private static function get_translatable_strings_from_form_settings( $form, &$string_data ) {
		$form_keys = self::get_form_strings( $form );

		// Add delete options.
		if ( $form->editable ) {
			$string_data['delete_msg'] = __( 'Your entry was successfully deleted', 'formidable' );
		}

		foreach ( $form_keys as $key ) {
			if ( isset( $form->{$key} ) && $form->{$key} != '' ) {
				$string_data[ $key ] = $form->{$key};
			} else if ( isset( $form->options[ $key ] ) && $form->options[ $key ] != '' && $form->options[ $key ] !== '[default-message]' ) {
				if ( is_array( $form->options[ $key ] ) ) {
					foreach ( $form->options[ $key ] as $k => $value ) {
						$string_data[ $key . '-' . $k ] = $value;
					}
				} else {
					$string_data[ $key ] = $form->options[ $key ];
				}
			}
		}

		// Add draft translations
		if ( isset( $form->options['save_draft'] ) && $form->options['save_draft'] ) {
			$string_data['draft_label'] = __( 'Save Draft', 'formidable' );
		}

		// Always update global strings in case they have changed.
		$frm_settings = FrmAppHelper::get_settings();
		if ( is_callable( array( $frm_settings, 'translatable_strings' ) ) ) {
			$register = new FrmWpmlRegister( compact( 'form' ) );
			foreach ( $frm_settings->translatable_strings() as $global_string ) {
				$string_data[ $global_string ] = $frm_settings->{$global_string};
				$register->update_string( $form->id . '-' . $global_string, $string_data[ $global_string ] );
			}
		} else {
			_deprecated_function( __FUNCTION__, '1.05', 'Please update Formidable Forms to the latest version.' );
			$string_data['invalid_msg'] = $frm_settings->invalid_msg;
		}

		self::add_registration_strings( $form, $string_data );
	}

	/**
	 * Get translatabe strings from the Registration add-on.
	 *
	 * @since 1.05
	 * @param object $form
	 * @param array  $string_data
	 * @return array
	 */
	private static function add_registration_strings( $form, &$string_data ) {
		if ( ! class_exists( 'FrmRegGlobalSettings' ) ) {
			return;
		}

		$registration_settings = new FrmRegGlobalSettings();
		if ( ! method_exists( $registration_settings, 'get_translatable_strings' ) ) {
			return;
		}

		$settings = FrmRegActionHelper::get_registration_settings_for_form( $form );
		if ( empty( $settings ) ) {
			// Don't add strings unless the form has a registration form action.
			return;
		}

		$keys   = $registration_settings->get_translatable_strings();
		$values = $registration_settings->get_global_messages();

		$register = new FrmWpmlRegister( compact( 'form' ) );

		foreach ( $keys as $key ) {
			$string_data[ 'reg_' . $key ] = $values[ $key ];
			$register->update_string( $form->id . '-reg_' . $key, $values[ $key ] );
		}
	}

	/**
	 * Get the strings that need translation for a single field
	 *
	 * @since 1.04
	 * @param object $field
	 * @param int $form_id
	 * @param boolean $is_repeating
	 * @param array $string_data
	 */
	private static function get_translatable_strings_for_single_field( $field, $form_id, $is_repeating, &$string_data ) {
		if ( $field->form_id != $form_id ) {
			// this field is in a repeating section
			return;
		}

		if ( $is_repeating && $field->type == 'end_divider' ) {
			self::add_translations_for_end_divider( $field, $string_data );
			return;
		}

		$keys = array(
			'name',
			'description',
			'default_value',
			'required_indicator',
			'blank',
			'unique_msg',
			'prev_value',
		);
		self::remove_unused_keys( $field, $keys );

		foreach ( $keys as $key ) {
			$string_name = 'field-' . $field->id . '-' . $key;

			if ( $key === 'default_value' ) {
				$default_value = self::get_default_value_for_translation( $field );
				if ( is_array( $default_value ) ) {
					foreach ( $default_value as $k => $v ) {
						if ( $v !== '' ) {
							$string_data[ $string_name . '-' . $k ] = $v;
						}
					}
				} elseif ( $default_value !== '' ) {
					$string_data[ $string_name ] = $default_value;
				}
			} else if ( isset( $field->{$key} ) && $field->{$key} != '' && ! is_array( $field->{$key} ) ) {
				$string_data[ $string_name ] = $field->{$key};
			} else if ( isset( $field->field_options[ $key ] ) && $field->field_options[ $key ] != '' && ! is_array( $field->field_options{$key} ) ) {
				$string_data[ $string_name ] = $field->field_options[ $key ];
			}
		}

		switch ( $field->type ) {
			case 'date':
				if ( isset( $field->field_options['locale'] ) ) {
					if ( $field->field_options['locale'] == '' ) {
						$field->field_options['locale'] = 'en';
					}
					$string_data[ 'field-' . $field->id . '-locale' ] = $field->field_options['locale'];
				}
				break;
			case 'address':
				foreach ( array( 'line1_desc', 'line2_desc', 'city_desc', 'state_desc', 'zip_desc', 'country_desc' ) as $address_text ) {
					$default_value = isset( $field->field_options[ $address_text ] ) ? $field->field_options[ $address_text ] : '';
					$string_data[ 'field-' . $field->id . '-' . $address_text ] = $default_value;
				}
				break;
			case 'email':
			case 'password':
				if ( isset( $field->field_options['conf_field'] ) && $field->field_options['conf_field'] ) {
					$string_data[ 'field-' . $field->id . '-conf_desc' ] = $field->field_options['conf_desc'];
				}

				// no break
			case 'url':
			case 'website':
			case 'phone':
			case 'image':
			case 'number':
			case 'file':
				if ( isset( $field->field_options['invalid'] ) && $field->field_options['invalid'] != '' ) {
					$string_data[ 'field-' . $field->id . '-invalid' ] = $field->field_options['invalid'];
				}
				break;
			case 'select':
			case 'checkbox':
			case 'radio':
				if ( is_array( $field->options ) && ! isset( $field->options['label'] ) ) {
					foreach ( $field->options as $index => $choice ) {
						if ( is_array( $choice ) ) {
							$choice = isset( $choice['label'] ) ? $choice['label'] : reset( $choice );
						}
						$string_data[ 'field-' . $field->id . '-choice-' . $choice ] = $choice;
					}
				} else {
					if ( is_array( $field->options ) ) {
						$field->options = isset( $field->options['label'] ) ? $field->options['label'] : reset( $field->options );
					}

					$string_data[ 'field-' . $field->id . '-choice-' . $field->options ] = $field->options;
				}
				break;
		}
	}

	/**
	 * Add translations for an end divider field
	 *
	 * @since 1.05
	 * @param object $field
	 * @param array $string_data
	 */
	private static function add_translations_for_end_divider( $field, &$string_data ) {
		$keys = array( 'add_label', 'remove_label' );

		foreach ( $keys as $key ) {
			$string_name = 'field-' . $field->id . '-' . $key;
			$string_data[ $string_name ] = $field->field_options[ $key ];
		}
	}
}
