<?php

class FrmWpmlSettingsController {

	public static function include_updater() {
		if ( class_exists( 'FrmAddon' ) ) {
			FrmWpmlUpdate::load_hooks();
		}
	}

	public static function load_lang() {
		$plugin_folder = FrmWpmlAppHelper::plugin_folder();
		load_plugin_textdomain( 'formidable-wpml', false, $plugin_folder . '/languages/' );
	}

	/**
	 * Tell WPML that we want formidable forms translated.
	 */
	public static function get_translatable_types( $types ) {
		$slug = 'formidable';
		$name = 'Formidable';

		if ( isset( $types[ $slug ] ) ) {
			return $types;
		}

		$type                = new stdClass();
		$type->name          = $slug;
		$type->label         = $name;
		$type->prefix        = 'package';
		$type->external_type = 1;

		$type->labels                = new stdClass();
		$type->labels->singular_name = $name;
		$type->labels->name          = $name;

		$types[ $slug ]              = $type;

		return $types;
	}

	public static function add_translate_button( $values ) {
		?>
		<a href="<?php echo esc_url( admin_url( 'admin.php?page=formidable&frm_action=translate&id=' . $values['id'] ) ); ?>" class="button-secondary frm-button-secondary">
		   <?php esc_html_e( 'Translate Form', 'formidable-wpml' ); ?>
	   </a>
		<?php
	}

	public static function translated() {
		//don't continue another action
		return true;
	}

	public static function update_translate() {
		$strings = FrmAppHelper::get_param( 'frm_wpml', '', 'post', 'sanitize_text_field' );
		if ( empty( $strings ) ) {
			self::translate();
			return;
		}

		$nonce = FrmAppHelper::get_param( 'frm_translate_form', '', 'post', 'sanitize_text_field' );
		if ( empty( $nonce ) || ! wp_verify_nonce( $nonce, 'frm_translate_form_nonce' ) ) {
			global $frm_settings;
			wp_die( esc_html( $frm_settings->admin_permission ) );
		}

		self::update_strings( $strings );

		$message = __( 'Translations Successfully Updated', 'formidable-wpml' );
		self::translate( $message );
	}

	/**
	 * Update any translations saved on the Translation page.
	 *
	 * @since 1.05
	 *
	 * @param array $strings The values POSTed from the translation form.
	 */
	private static function update_strings( $strings ) {
		global $wpdb;

		foreach ( $strings as $tkey => $t ) {
			$st = array(
				'value'  => $t['value'],
				'status' => ( isset( $t['status'] ) ) ? $t['status'] : ICL_STRING_TRANSLATION_NOT_TRANSLATED,
			);

			if ( is_numeric( $tkey ) ) {
				$wpdb->update( $wpdb->prefix . 'icl_string_translations', $st, array( 'id' => $tkey ) );
			} elseif ( ! empty( $t['value'] ) ) {
				$info = explode( '_', $tkey );
				if ( ! is_numeric( $info[0] ) ) {
					continue;
				}

				$id = $wpdb->get_var(
					$wpdb->prepare(
						'SELECT id FROM ' . $wpdb->prefix . 'icl_string_translations WHERE language=%s AND string_id=%d',
						$info[1],
						$info[0]
					)
				);

				if ( ! empty( $id ) ) {
					$wpdb->update( $wpdb->prefix . 'icl_string_translations', $st, array( 'id' => $id ) );
				} else {
					// Create translations that don't exist.
					$string_id = $info[0];
					$language  = $info[1];
					icl_add_string_translation( $string_id, $language, $st['value'], $st['status'] );
				}
			}
			unset( $t, $tkey );
		}
	}

	public static function translate( $message = '' ) {
		if ( ! function_exists( 'icl_t' ) ) {
			esc_html_e( 'You do not have WPML installed', 'formidable-wpml' );
			return;
		}

		global $sitepress;

		$id           = FrmAppHelper::get_param( 'id', false, 'get', 'absint' );
		$form         = FrmForm::getOne( $id );
		$fields_array = FrmField::getAll( array( 'fi.form_id' => $id ), 'field_order' );

		$fields   = array();
		$form_ids = array();

		foreach ( $fields_array as $f ) {
			$fields[ $f->id ] = $f;
			$form_ids[ $f->form_id ] = absint( $f->form_id );
			unset( $f );
		}
		unset( $fields_array );

		$langs = $sitepress->get_active_languages();
		$default_language = self::get_string_language();
		ksort( $langs );

		$col_order = array( $default_language );
		foreach ( $langs as $lang ) {
			if ( $lang['code'] == $default_language ) {
				continue;
			}

			$col_order[] = $lang['code'];
		}

		$lang_count = ( count( $langs ) - 1 );

		// Register any missing strings.
		self::get_translatable_items( array(), 'formidable' );

		$strings      = self::get_strings_for_form( $form_ids );
		$translations = self::get_string_translations( $strings );
		$register     = new FrmWpmlRegister( compact( 'form' ) );

		include( FrmWpmlAppHelper::plugin_path() . '/views/translate.php' );
	}

	/**
	 * Get all registered strings for the current form.
	 *
	 * @since 1.05
	 *
	 * @param array $form_ids
	 * @return array
	 */
	private static function get_strings_for_form( $form_ids ) {
		global $wpdb;

		$current_id = FrmAppHelper::get_param( 'id', false, 'get', 'absint' );
		$query_args = array( 'formidable' );
		$like       = '';
		foreach ( $form_ids as $form_id ) {
			if ( ! empty( $like ) ) {
				$like .= ' OR ';
			}
			$like .= 'name LIKE %s';

			// If this is a child form, only get the field values.
			$query_args[] = $form_id . ( $form_id == $current_id ? '' : '\_field-' ) . '\_%';
		}

		// phpcs:ignore WordPress.DB.PreparedSQL
		$query = $wpdb->prepare( "SELECT id, name, value, language FROM {$wpdb->prefix}icl_strings WHERE context=%s AND (" . $like . ') ORDER BY name DESC', $query_args );
		return $wpdb->get_results( $query, OBJECT_K ); // phpcs:ignore WordPress.DB.PreparedSQL
	}

	/**
	 * Get all translations for the registered strings.
	 *
	 * @since 1.05
	 */
	private static function get_string_translations( $strings ) {
		global $wpdb;

		if ( empty( $strings ) ) {
			return array();
		}

		$where = array( 'string_id' => array_keys( $strings ) );
		$args  = array( 'order_by' => 'language ASC' );

		return FrmDb::get_results( $wpdb->prefix . 'icl_string_translations', $where, 'id, string_id, value, status, language', $args );
	}

	/**
	 * Show the input box on the translation page for a single string.
	 *
	 * @since 1.05
	 * @param array $atts
	 */
	public static function include_single_input( $atts ) {
		$name = 'frm_wpml[' . $atts['input_id'] . ']';
		if ( $atts['value'] === '' || ! isset( $atts['complete'] ) ) {
			$atts['complete'] = 0;
		}
		include( FrmWpmlAppHelper::plugin_path() . '/views/single-input.php' );
	}

	/**
	 * Pick from available options for the datepicker locale.
	 *
	 * @since 1.05
	 * @param object $string - The string to be translated.
	 * @param array  $atts - Includes the form id and array of all fields in the form.
	 */
	public static function maybe_get_translation_options( $string, $atts ) {
		$options   = array();
		$is_locale = strpos( $string->name, '-locale' ) !== false;
		$is_field  = strpos( $string->name, $atts['id'] . '_field-' ) === 0;
		if ( ! $is_locale || ! $is_field ) {
			return $options;
		}

		// Get the field id from the string name.
		$fid      = explode( '-', str_replace( $atts['id'] . '_field-', '', $string->name ), 2 );
		$field_id = $fid[0];
		if ( ! isset( $atts['fields'][ $field_id ] ) ) {
			return $options;
		}

		$field_type = $atts['fields'][ $field_id ]->type;
		if ( $field_type === 'date' ) {
			$options = FrmAppHelper::locales( 'date' );
		}

		return $options;
	}

	/**
	 * Show the locale dropdown on the translation page.
	 *
	 * @since 1.05
	 * @param string $input_name
	 * @param array  $options
	 * @param string $selected
	 */
	public static function show_dropdown_options( $input_name, $options, $selected = '' ) {
		?>
		<select name="<?php echo esc_attr( $input_name ); ?>[value]">
			<option value=""> </option>
			<?php foreach ( $options as $code => $label ) { ?>
				<option value="<?php echo esc_attr( $code ); ?>" <?php selected( $selected, $code ); ?>>
					<?php echo esc_html( $label ); ?>
				</option>
			<?php } ?>
		</select>
		<?php
	}

	public static function get_string_language() {
		global $sitepress;

		$string_version = defined( 'WPML_ST_VERSION' ) ? WPML_ST_VERSION : 1;
		if ( class_exists( 'WPML_Language_Of_Domain' ) ) {
			$lang_of_domain = new WPML_Language_Of_Domain( $sitepress );
			$default_language = $lang_of_domain->get_language( 'formidable' );
			if ( ! $default_language ) {
				$default_language = FrmWpmlAppHelper::get_default_language();
			}
		} elseif ( version_compare( $string_version, '2.2.5', '>' ) ) {
			$default_language = 'en';
		} else {
			global $sitepress_settings;
			$default_language = ! empty( $sitepress_settings['st']['strings_language'] ) ? $sitepress_settings['st']['strings_language'] : FrmWpmlAppHelper::get_default_language();
		}

		return $default_language;
	}

	/**
	 * Update the saved ICL strings
	 *
	 * @since 1.05
	 * @param int $form_id
	 * @param array $values
	 */
	public static function update_saved_wpml_strings( $form_id, $values ) {
		$register = new FrmWpmlRegister( array( 'form' => $form_id ) );
		$register->update_form_fields( $values );
	}

	public static function delete_frm_wpml( $id ) {
		global $wpdb;

		//delete strings before a field is deleted
		$strings = $wpdb->get_col(
			$wpdb->prepare(
				"SELECT name FROM {$wpdb->prefix}icl_strings WHERE context=%s AND name LIKE %s",
				'formidable',
				"%_field-{$id}-%"
			)
		);

		if ( $strings ) {
			$register = new FrmWpmlRegister();
			$register->unregister( $strings );
		}
	}

	public static function get_translatable_items() {
		global $wpdb;

		$action  = FrmAppHelper::simple_get( 'frm_action', 'sanitize_title' );
		$form_id = FrmAppHelper::simple_get( 'id', 'absint' );

		if ( $action === 'translate' && ! empty( $form_id ) ) {
			$forms = FrmForm::getAll( $wpdb->prepare( 'parent_form_id=%d or id=%d', $form_id, $form_id ) );
		} else {
			$forms = FrmForm::getAll( "is_template=0 AND (status is NULL OR status = '' OR status = 'published')", ' ORDER BY name' );
		}

		foreach ( $forms as $form ) {

			$register = new FrmWpmlRegister( compact( 'form' ) );

			$strings = FrmWpmlString::form_strings( $form );

			// Register the strings with WPML.
			foreach ( $strings as $key => $value ) {
				$key = $form->id . '_' . $key;
				$key = ( function_exists( 'mb_substr' ) ) ? mb_substr( $key, 0, 160 ) : substr( $key, 0, 160 );
				if ( ! is_array( $value ) && ! icl_st_is_registered_string( 'formidable', $key ) ) {
					$register->register( $key, $value );
				}

				unset( $key, $value );
			}
		}
	}
}
